// Pulse Position Modulation
// (c) 2007, Idle Loop Software Design, LLC
//
// contact: robotics@idleloop.com
//
// This code is supplied as sample code and may be freely used,
// without any sort of warrantee whatsoever. You've been warned.
//
// The following is intended to decode the signal coming out of a
// Vex R/C receiver, but should work for any PPM signal.
//
// This is written for an Atmel ATmega processor.
//
// You'll need to supply a couple of timer functions. See comments below

#include <avr/io.h>
#include <avr/interrupt.h>

#include "PPM.h"

// host must supply a timer with sufficient resolution for measuring the
// pulse widths
//
// the following timer functions must be supplied
ulong CtickSecond(); // returns number of counts per second
ulong Tick(); // the current timer count

struct PWMR
{
	ulong cPulse;
	uint ctick;
	uchar fNew;
};
static PWMR s_apwmr[cchannel];

#define fpin0   0x01
#define fpin1   0x02
#define fpin2   0x04
#define fpin3   0x08
#define fpin4   0x10
#define fpin5   0x20
#define fpin6   0x40
#define fpin7   0x80

// use port D pin 7
#define grfpinMask  fpin7
#define regPORT     PORTD
#define regPIN      PIND
#define regDDR      DDRD
#define regPCMSK    PCMSK3
#define bitPCE      (1 << PCIE3)
#define vectIntPC   PCINT3_vect

static uint s_ctickPulseMin;
static uint s_ctickGapMin;
static uint s_ctickResetMin;

static int s_ctickPWMLow;
static int s_ctickPWMRange;

static ulong s_tickUp, s_tickPrev;
static int s_channelCur = -1;
static ulong s_dtickPrev = 0;

static ulong s_atickRecord[20];
static int s_itickRecord = 0;

// call this function once at the start of your program
// before calling any other PPM functions
void StartPPMDecoder()
{
	ulong ctickSecond = CtickSecond();
	s_ctickPulseMin = (ctickSecond * 3 + 5000) / 10000;
	s_ctickGapMin = (ctickSecond * 2 + 5000) / 10000;
	s_ctickResetMin = (ctickSecond * 40 + 5000) / 10000;

	// NOTE: these values are tuned for (my) Vex radio
	s_ctickPWMLow = (ctickSecond*955 + 500000UL) / 1000000UL;
	s_ctickPWMRange = (ctickSecond*1089 + 500000UL) / 1000000UL;
	
	s_tickUp = s_tickPrev = Tick();

	regDDR &= ~grfpinMask;   // set pin(s) for input
	regPORT |= grfpinMask;   // Vex Receiver needs a pull-up resister
	regPCMSK |= grfpinMask;  // enable PCINTX for port A
	PCICR |= bitPCE; // enable PCIE pin change interrupt enable 0 for the A port pins
}

// this is the interrupt handler that does all the
// heavy lifting
ISR(vectIntPC)
{
	uchar grfpinDCur = regPIN & grfpinMask;

	ulong tickCur = Tick();
	ulong dtick = tickCur - s_tickPrev;
	
	s_atickRecord[s_itickRecord] = dtick;
	if (++s_itickRecord >= (int)DIM(s_atickRecord))
		s_itickRecord = 0;

	if (grfpinDCur)
	{
		// rising edge
		if (dtick >= s_ctickPulseMin)
		{
			// we got the reset pulse, ready for first channel
			if (dtick >= s_ctickResetMin)
			{
				s_channelCur = -1;
			}
			// got a channel pulse, save it if we're in a good state
			else if (0 <= s_channelCur && s_channelCur < (int)DIM(s_apwmr))
			{
				s_apwmr[s_channelCur].ctick = tickCur - s_tickUp;
				++s_apwmr[s_channelCur].cPulse;
				s_apwmr[s_channelCur].fNew = fTrue;
				s_dtickPrev = s_apwmr[s_channelCur].ctick;
			}

			s_tickUp = tickCur;
		}
		else
		{
			// pulse width too small, block until next reset pulse
			s_channelCur = DIM(s_apwmr);
		}
	}
	else
	{
		// falling edge
		if (dtick >= s_ctickGapMin)
		{
			// we got a good gap pulse, increment channel if in good state
			if (s_channelCur < (int)DIM(s_apwmr))
				++s_channelCur;
		}
		else
		{
			// pulse width too small, block until next reset pulse
			s_channelCur = DIM(s_apwmr);
		}
	}
	
	s_tickPrev = tickCur;
}

// get the current read for the specified channel
// returns a value in the range [0-255].
uchar PwmGetChannel(CHAN channel)
{
	char sregSav;
	long ctick;
	int fNew;

	sregSav = SREG;	// cache register holding Global Interrupt Flag
	cli();			// clear Global Interrupt Flag
	ctick = s_apwmr[channel].ctick;
	fNew = s_apwmr[channel].fNew;
	s_apwmr[channel].fNew = fFalse;
	SREG = sregSav;	// restore register holding Global Interrupt Flag
	
	ctick = ((ctick - s_ctickPWMLow) * 256 + s_ctickPWMRange/2) / s_ctickPWMRange;
	
	if (!fNew)
		ctick = 127;
	
	if (ctick < 0)
		ctick = 0;
	else if (ctick > 255)
		ctick = 255;
	
	return (uchar)ctick;
}

// call this function to see if there's been an update for
// the specified channel since the last time PwmGetChannel
// was called for that channel.
int FChannelUpdate(CHAN channel)
{
	char sregSav;
	int fNew;

	sregSav = SREG;	// cache register holding Global Interrupt Flag
	cli();			// clear Global Interrupt Flag
	fNew = s_apwmr[channel].fNew;
	SREG = sregSav;	// restore register holding Global Interrupt Flag
	
	return fNew;
}
